//
//  ImpChannel.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/4/2.
//

#ifndef ImpChannel_h
#define ImpChannel_h


#endif /* ImpChannel_h */

#import <Foundation/Foundation.h>

/**
 * @class ImpChannel
 * @brief A class for managing a communication channel with a queue and synchronization.
 * @ingroup JSONMng
 */
@interface ImpChannel : NSObject {
    NSMutableArray *dataQueue;  /**< The queue used to store data. */
    NSCondition *condition;     /**< The condition variable used for synchronization. */
    BOOL completed;             /**< A flag indicating whether the channel is completed. */
    NSUInteger capacity;        /**< The capacity limit of the channel. */
    NSString *channelId;        /**< The ID of the channel. */
}

// 方法声明
/**
 * @brief Sends a message to the channel.
 *
 * @param msg The message to be sent.
 */
- (void)send:(id)msg;

/**
 * @brief Receives a message from the channel.
 *
 * @return The received message.
 */
- (id)receive;

/**
 * @brief Marks the channel as complete.
 */
- (void)complete;

/**
 * @brief Resets the channel to its initial state.
 */
- (void)reset;

/**
 * @brief Checks if the channel is empty.
 *
 * @return YES if the channel is empty, NO otherwise.
 */
- (BOOL)isEmpty;

/**
 * @brief Gets the message at the front of the channel without removing it.
 *
 * @return The message at the front of the channel.
 */
- (id)front;

/**
 * @brief Removes the message at the front of the channel.
 */
- (void)pop;

/**
 * @brief Waits until a message is available to read.
 *
 * @return YES if a message is available, NO otherwise.
 */
- (BOOL)waitToRead;

/**
 * @brief Waits for a message to be available to read with a timeout.
 *
 * @param timeout The timeout interval.
 * @return YES if a message is available within the timeout, NO otherwise.
 */
- (BOOL)waitToReadWithTimeout:(NSTimeInterval)timeout;

/**
 * @brief Tries to read a message from the channel without blocking.
 *
 * @param msg A pointer to the message that will be read.
 * @return YES if a message was read, NO otherwise.
 */
- (BOOL)tryRead:(id *)msg;

/**
 * @brief Sets the capacity of the channel.
 *
 * @param newCapacity The new capacity to be set.
 */
- (void)setCapacity:(NSUInteger)newCapacity;

/**
 * @brief Gets the message at a specific index in the channel.
 *
 * @param index The index of the message to be retrieved.
 * @return The message at the specified index.
 */
- (id)getAt:(NSUInteger)index;

/**
 * @brief Gets the current size of the channel.
 *
 * @return The size of the channel.
 */
- (NSUInteger)size;

/**
 * @brief Gets the ID of the channel.
 *
 * @return The ID of the channel.
 */
- (NSString *)getId;

/**
 * @brief Sets the channel to be empty.
 */
- (void)setEmpty;

@end

