//
//  MaxicodeBarcode.h
//  sdk_json_Objective-C
//
//  Created by Realbuber on 2024/5/16.
//

#ifndef MaxicodeBarcode_h
#define MaxicodeBarcode_h

#import <Foundation/Foundation.h>
#import <UniPRT/AMaxicodeBarcode.h>
#import <UniPRT/IRotation.h>
#import <UniPRT/IPoint.h>
#import <UniPRT/IMaxicodeData.h>

/**
 * \~English
 * @class MaxicodeBarcode
 * @brief A class representing a Maxicode barcode for TSPL.
 *
 * This class inherits from AMaxicodeBarcode and implements the IRotation protocol,
 * providing functionality for initializing and describing Maxicode barcodes in TSPL.
 *
 * @ingroup TSPL
 *
 * \~Chinese
 * @class MaxicodeBarcode
 * @brief 表示用于 TSPL 的 Maxicode 条码的类。
 *
 * 该类继承自 AMaxicodeBarcode 并实现了 IRotation 协议，
 * 提供了在 TSPL 中初始化和描述 Maxicode 条码的功能。
 *
 * @ingroup TSPL
 *
 * \~Chinese-Traditional
 * @class MaxicodeBarcode
 * @brief 表示用於 TSPL 的 Maxicode 條碼的類。
 *
 * 該類繼承自 AMaxicodeBarcode 並實現了 IRotation 協議，
 * 提供了在 TSPL 中初始化和描述 Maxicode 條碼的功能。
 *
 * @ingroup TSPL
 */
@interface MaxicodeBarcode : AMaxicodeBarcode <IRotation>

/**
 * \~English
 * @brief The starting point of the barcode.
 *
 * \~Chinese
 * @brief 条码的起始点。
 *
 * \~Chinese-Traditional
 * @brief 條碼的起始點。
 */
@property (nonatomic, strong) id<IPoint> start;

/**
 * \~English
 * @brief The data to be encoded in the barcode.
 *
 * \~Chinese
 * @brief 要在条码中编码的数据。
 *
 * \~Chinese-Traditional
 * @brief 要在條碼中編碼的數據。
 */
@property (nonatomic, strong) id<IMaxicodeData> data;

/**
 * \~English
 * @brief The rotation of the barcode.
 *
 * \~Chinese
 * @brief 条码的旋转角度。
 *
 * \~Chinese-Traditional
 * @brief 條碼的旋轉角度。
 */
@property (nonatomic, assign) RotateEnum rotation;

/**
 * \~English
 * @brief Zipper pattern can be used to help detect printing defects e.g. light/dark printing.
 *
 * \~Chinese
 * @brief 拉链图案可用于帮助检测打印缺陷，例如浅色/深色打印。
 *
 * \~Chinese-Traditional
 * @brief 拉鍊圖案可用於幫助檢測打印缺陷，例如淺色/深色打印。
 */
@property (nonatomic, assign) BOOL zipperPattern;

/**
 * \~English
 * @brief Initializes a new instance of the MaxicodeBarcode class with the specified start point and data.
 *
 * @param start The starting point of the barcode.
 * @param data The data to be encoded in the barcode.
 * @return A new instance of MaxicodeBarcode.
 *
 * \~Chinese
 * @brief 使用指定的起始点和数据初始化 MaxicodeBarcode 类的新实例。
 *
 * @param start 条码的起始点。
 * @param data 要在条码中编码的数据。
 * @return MaxicodeBarcode 的新实例。
 *
 * \~Chinese-Traditional
 * @brief 使用指定的起始點和數據初始化 MaxicodeBarcode 類的新實例。
 *
 * @param start 條碼的起始點。
 * @param data 要在條碼中編碼的數據。
 * @return MaxicodeBarcode 的新實例。
 */
- (instancetype)initWithStart:(id<IPoint>)start data:(id<IMaxicodeData>)data;

/**
 * \~English
 * @brief Returns a string that can be sent to the printer.
 *
 * This method generates a string representation of the MaxicodeBarcode instance that
 * is formatted and ready to be sent to a printer. The string includes all the
 * necessary commands and data required for printing in TSPL.
 *
 * @return A string description of the MaxicodeBarcode instance.
 *
 * \~Chinese
 * @brief 返回可以发送到打印机的字符串。
 *
 * 该方法生成 MaxicodeBarcode 实例的字符串表示形式，
 * 格式化并准备好发送到打印机。字符串包括在 TSPL 中打印所需的所有命令和数据。
 *
 * @return MaxicodeBarcode 实例的字符串描述。
 *
 * \~Chinese-Traditional
 * @brief 返回可以發送到打印機的字符串。
 *
 * 該方法生成 MaxicodeBarcode 實例的字符串表示形式，
 * 格式化並準備好發送到打印機。字符串包括在 TSPL 中打印所需的所有命令和數據。
 *
 * @return MaxicodeBarcode 實例的字符串描述。
 */
- (NSString *)description;

@end

#endif /* MaxicodeBarcode_h */
