//
//  btComm.h
//  sdk_json_Objective-C_IOS
//
//  Created by Realbuber on 2024/4/23.
//

/** @cond */

#ifndef btComm_h
#define btComm_h


#endif /* btComm_h */
#import <Foundation/Foundation.h>
#import <UniPRT/AComm.h>
#import <ExternalAccessory/ExternalAccessory.h>

/**
 * @class BtComm
 * @ingroup Comm
 *
 * \~English
 * @brief A class for Bluetooth communication, inheriting from AComm.
 *
 * \~Chinese
 * @brief 一个用于蓝牙通信的类，继承自 AComm。
 *
 * \~Chinese-Traditional
 * @brief 一個用於藍牙通信的類別，繼承自 AComm。
 */
@interface BtComm : AComm <NSStreamDelegate>

/** @cond */
@property (strong, nonatomic) NSInputStream *inputStream;
@property (strong, nonatomic) NSOutputStream *outputStream;
/** @endcond */

@property (nonatomic, readwrite) BOOL connected;

/**
 * \~English
 * @brief Initializes a new instance with the given device address.
 *
 * @param deviceAddress The Bluetooth address of the device.
 * @return An initialized instance of BtComm, or nil if the object could not be created.
 *
 * \~Chinese
 * @brief 使用给定的设备地址初始化一个新实例。
 *
 * @param deviceAddress 设备的蓝牙地址。
 * @return 一个初始化的 BtComm 实例，如果无法创建对象，则返回 nil。
 *
 * \~Chinese-Traditional
 * @brief 使用給定的設備地址初始化一個新實例。
 *
 * @param deviceAddress 設備的藍牙地址。
 * @return 一個初始化的 BtComm 實例，如果無法創建對象，則返回 nil。
 */
- (instancetype)initWithDeviceAddress:(NSString *)deviceAddress;

/**
 * \~English
 * @brief Opens the connection to the Bluetooth device.
 *
 * \~Chinese
 * @brief 打开与蓝牙设备的连接。
 *
 * \~Chinese-Traditional
 * @brief 打開與藍牙設備的連接。
 */
- (void)open;

/**
 * \~English
 * @brief Closes the connection to the Bluetooth device.
 *
 * \~Chinese
 * @brief 关闭与蓝牙设备的连接。
 *
 * \~Chinese-Traditional
 * @brief 關閉與藍牙設備的連接。
 */
- (void)close;

/**
 * \~English
 * @brief Writes data to the Bluetooth device.
 *
 * @param text The data to write to the device.
 *
 * \~Chinese
 * @brief 将数据写入蓝牙设备。
 *
 * @param text 要写入设备的数据。
 *
 * \~Chinese-Traditional
 * @brief 將數據寫入藍牙設備。
 *
 * @param text 要寫入設備的數據。
 */
- (void)write:(NSData *)text;

/**
 * \~English
 * @brief Reads data from the Bluetooth device.
 *
 * @return The data read from the device.
 *
 * \~Chinese
 * @brief 从蓝牙设备读取数据。
 *
 * @return 从设备读取的数据。
 *
 * \~Chinese-Traditional
 * @brief 從藍牙設備讀取數據。
 *
 * @return 從設備讀取的數據。
 */
- (NSData *)read;

/**
 * \~English
 * @brief Writes data to the Bluetooth device and waits for a response.
 *
 * @param text The data to write to the device.
 * @param responseStartTimeOut The timeout interval for the start of the response.
 * @param responseEndTimeOut The timeout interval for the end of the response.
 * @param completionToken A token indicating the completion of the response.
 * @return The data received in response from the device.
 *
 * \~Chinese
 * @brief 将数据写入蓝牙设备并等待响应。
 *
 * @param text 要写入设备的数据。
 * @param responseStartTimeOut 响应开始的超时间隔。
 * @param responseEndTimeOut 响应结束的超时间隔。
 * @param completionToken 表示响应完成的令牌。
 * @return 从设备接收到的响应数据。
 *
 * \~Chinese-Traditional
 * @brief 將數據寫入藍牙設備並等待響應。
 *
 * @param text 要寫入設備的數據。
 * @param responseStartTimeOut 響應開始的超時間隔。
 * @param responseEndTimeOut 響應結束的超時間隔。
 * @param completionToken 表示響應完成的令牌。
 * @return 從設備接收到的響應數據。
 */
- (NSData *)writeAndWaitForResponse:(NSData *)text
                 responseStartTimeOut:(NSTimeInterval)responseStartTimeOut
                   responseEndTimeOut:(NSTimeInterval)responseEndTimeOut
                      completionToken:(NSString *)completionToken;

@end

/** @endcond */
