//
//  JsonMng.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/4/3.
//

#ifndef JsonMng_h
#define JsonMng_h


#endif /* JsonMng_h */


#import <Foundation/Foundation.h>
#import <AutoIdSDK/JsonStringTokenizer.h>
#import <AutoIdSDK/JsonStringBuilder.h>
#import <AutoIdSDK/ImpChannel.h>
#import <AutoIdSDK/CancellationToken.h>
#import <AutoIdSDK/TcpComm.h>

/**
 * @class JsonComm
 * @brief A class for managing JSON-based communication.
 * @ingroup JSONComm
 */


@interface JsonComm : NSObject {
    BOOL disposed;                                      /**< A flag indicating whether the object is disposed. */
    NSLock *_listenerMutex;                             /**< Mutex for synchronizing listener access. */
    NSMutableDictionary *_SolicitedListenerTable;       /**< Table of solicited listeners. */
    NSMutableArray *_msgListeners_Solicited;            /**< List of solicited message listeners. */
    CancellationToken *_cancellationTokenSource;        /**< Cancellation token source. */
    JsonStringTokenizer *_jsonStringTokenizer;          /**< JSON string tokenizer. */
    NSThread *_asyncJsonMsgTransceiverThread;           /**< Thread for asynchronous JSON message transceiver. */
    
    BOOL _Connected;                                    /**< A flag indicating whether the connection is established. */
    NSMutableArray *_msgListeners_AllMsgs;              /**< List of all message listeners. */
    NSMutableArray *_msgListeners_Unsolicited;          /**< List of unsolicited message listeners. */
    NSMutableString *_jsonReceived;                     /**< Buffer for received JSON data. */
    NSLock *_semaphoreListeners;                        /**< Semaphore for synchronizing listener access. */
    
    // id _commToPtr;                                   /**< Pointer to the communication instance (commented out). */
    NSInteger _iCommTyp;                                /**< Type of communication. */
}

/**
 * @enum CommType
 * @brief Enumeration for communication types.
 */
typedef NS_ENUM(NSInteger, CommType) {
    USB_COMM,      /**< USB communication. */
    BT_COMM,       /**< Bluetooth communication. */
    TCP_COMM       /**< TCP communication. */
};

@property (strong, nonatomic) id commToPtr;                                           /**< Pointer to the communication instance. */
@property (assign, nonatomic) CommType commType;                                      /**< The type of communication. */
@property (assign, nonatomic) BOOL m_ThIsRunning;                                     /**< A flag indicating whether the thread is running. */
@property (nonatomic, strong) NSMutableArray *msgListeners_Solicited;                 /**< List of solicited message listeners. */
@property (nonatomic, strong) NSMutableArray *msgListeners_Unsolicited;               /**< List of unsolicited message listeners. */
@property (nonatomic, strong) NSMutableArray *msgListeners_AllMsgs;                   /**< List of all message listeners. */
@property (nonatomic, strong) NSMutableDictionary<NSString *, ImpChannel *> *solicitedListenerTable; /**< Table of solicited listeners. */
@property (nonatomic, strong) ImpChannel *msgOut;                                     /**< Channel for outgoing messages. */
@property (nonatomic, strong) ImpChannel *MonitorMsgOut;                              /**< Channel for monitoring outgoing messages. */
@property (nonatomic, strong) ImpChannel *MsgsFromPtr;                                /**< Channel for messages from the pointer. */
@property (nonatomic, assign) BOOL usingDataPort;                                     /**< A flag indicating whether the data port is used. */
@property (nonatomic, strong) NSString *solicitedUnsolicitedMsg;                      /**< Message for solicited/unsolicited communication. */
@property (nonatomic, assign) NSInteger iMsgCount;                                    /**< Message count. */

/**
 * @enum ChannelType
 * @brief Enumeration for channel types.
 */
typedef NS_ENUM(NSInteger, ChannelType) {
    SOLICITED_MSGS_ONLY,  /**< Only solicited messages. */
    UNSOLICITED_MSGS_ONLY, /**< Only unsolicited messages. */
    ALL_MSGS              /**< All messages. */
};

/**
 * @brief Disposes the object, releasing any resources.
 */
- (void)dispose;

/**
 * @brief Disposes the object with an optional flag.
 *
 * @param disposing A flag indicating whether to dispose managed resources.
 */
- (void)disposeWithFlag:(BOOL)disposing;

/**
 * @brief Disconnects a listener channel.
 *
 * @param chnlToRemove The channel to remove.
 */
- (void)listenerChannelDisconnect:(ImpChannel *)chnlToRemove;

/**
 * @brief Starts the JSON communication.
 */
- (void)start;

/**
 * @brief Stops the JSON communication.
 */
- (void)stop;

/**
 * @brief Extended stop function for the JSON communication.
 */
- (void)stopEx;

/**
 * @brief The task for transceiving JSON messages.
 */
- (void)jsonMsgTransceiverTask;

/**
 * @brief Gets the key for a solicited channel.
 *
 * @param chnlToLookFor The channel to look for.
 * @return The key for the solicited channel.
 */
- (NSString *)solicitedChnlKey:(ImpChannel *)chnlToLookFor;

/**
 * @brief Initializes a new instance with the given communication pointer and type.
 *
 * @param commToPtr The pointer to the communication instance.
 * @param commType The type of communication.
 * @return An initialized instance of JsonComm, or nil if the object could not be created.
 */
- (instancetype)initWithComm:(id)commToPtr commType:(NSInteger)commType;

/**
 * @brief Gets a new listener channel with specified capacity and type.
 *
 * @param capacity The capacity of the channel.
 * @param channelType The type of the channel.
 * @param solicitedMsgId The ID of the solicited message.
 * @return A new ImpChannel instance.
 */
- (ImpChannel *)listenerChannelGetNewWithCapacity:(NSInteger)capacity
                                      channelType:(ChannelType)channelType
                                   solicitedMsgId:(NSString *)solicitedMsgId;

/**
 * @brief Sends a management message.
 *
 * @param mgmtMsgToSend The management message to send.
 */
- (void)send:(NSString *)mgmtMsgToSend;

/**
 * @brief Gets an ImpChannel instance.
 *
 * @return An ImpChannel instance.
 */
- (ImpChannel *)getImpChannel;

/**
 * @brief Gets the TCP communication instance.
 *
 * @return A pointer to the TcpComm instance.
 */
- (TcpComm *)getPtrComm;

/**
 * @brief Sends a message and waits for a response.
 *
 * @param mgmtMsg The management message to send.
 * @param respTimeoutMs The response timeout in milliseconds.
 * @return The response message.
 */
- (NSString *)sendAndWaitForResponse:(NSString *)mgmtMsg timeout:(int)respTimeoutMs;

@end

