//
//  PrinterMonitor.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/4/11.
//

#ifndef PrinterMonitor_h
#define PrinterMonitor_h


#endif /* PrinterMonitor_h */

#import <Foundation/Foundation.h>
#import <AutoIdSDK/TcpComm.h>
#import <AutoIdSDK/JsonComm.h>
#import <AutoIdSDK/PrinterInfo.h>
#import <AutoIdSDK/SynchronizationContext.h>
#import <AutoIdSDK/JsonCommController.h>
#import <AutoIdSDK/JsonStringBuilder.h>


/**
 * @class PrinterMonitor
 * @brief A class for monitoring printer status.
 * @ingroup Monitor
 */
@interface PrinterMonitor : NSObject

/**
 * @brief The current engine status of the printer.
 */
@property (nonatomic, strong) NSString *engineStatus;

/**
 * @brief The current alert statuses of the printer.
 */
@property (nonatomic, strong) NSArray<NSString *> *alertStatus;

/**
 * @brief The current display statuses of the printer.
 */
@property (nonatomic, strong) NSArray<NSString *> *displayStatus;

/**
 * @brief A constant ID that the class does not care about.
 */
@property (nonatomic, strong) NSString *DONT_CARE_ID;

/**
 * @brief A flag indicating whether engine status listening is enabled.
 */
@property (nonatomic) BOOL engineStatusListening;

/**
 * @brief A flag indicating whether display status listening is enabled.
 */
@property (nonatomic) BOOL displayStatusListening;

/**
 * @brief A flag indicating whether alert status listening is enabled.
 */
@property (nonatomic) BOOL alertStatusListening;

/**
 * @brief Initializes a new instance with the given TCP communication object.
 *
 * @param connection The TCP communication object.
 * @return An initialized instance of PrinterMonitor, or nil if the object could not be created.
 */
- (instancetype)initWithTcpComm:(TcpComm *)connection;

/**
 * @brief Initializes a new instance with the given JSON communication object.
 *
 * @param mgmtComm The JSON communication object.
 * @return An initialized instance of PrinterMonitor, or nil if the object could not be created.
 */
- (instancetype)initWithJsonComm:(JsonComm *)mgmtComm;

/**
 * @brief Initializes a new instance with the given communication descriptor.
 *
 * @param commDescriptor The communication descriptor.
 * @return An initialized instance of PrinterMonitor, or nil if the object could not be created.
 */
- (instancetype)initWithCommDescriptor:(NSString *)commDescriptor;

/**
 * @brief Gets the printer information.
 *
 * @return A PrinterInfo object containing the printer information.
 */
- (PrinterInfo *)getPrinterInfo;

/**
 * @brief Gets the engine status of the printer.
 *
 * @return A string representing the engine status.
 */
- (NSString *)getEngineStatus;

/**
 * @brief Gets the fault status of the printer.
 *
 * @return An array of strings representing the fault status.
 */
- (NSArray<NSString *> *)getFaultStatus;

/**
 * @brief Sets whether engine status listening is enabled.
 *
 * @param value YES to enable listening, NO to disable.
 */
- (void)SetEngineStatusListening:(BOOL)value;

/**
 * @brief Sets whether display status listening is enabled.
 *
 * @param value YES to enable listening, NO to disable.
 */
- (void)SetDisplayStatusListening:(BOOL)value;

/**
 * @brief Sets whether alert status listening is enabled.
 *
 * @param value YES to enable listening, NO to disable.
 */
- (void)SetAlertStatusListening:(BOOL)value;

/**
 * @typedef EngineStatusCallback
 * @brief A callback type for engine status changes.
 *
 * @param newEngineState The new engine state.
 */
typedef void (^EngineStatusCallback)(NSString *newEngineState);

/**
 * @typedef DisplayStatusCallback
 * @brief A callback type for display status changes.
 *
 * @param panelText The new panel text.
 */
typedef void (^DisplayStatusCallback)(NSArray<NSString *> *panelText);

/**
 * @typedef AlertStatusCallback
 * @brief A callback type for alert status changes.
 *
 * @param alertText The new alert text.
 */
typedef void (^AlertStatusCallback)(NSArray<NSString *> *alertText);

/**
 * @brief The callback for engine status changes.
 */
@property (nonatomic, copy) EngineStatusCallback engineStatusCallback;

/**
 * @brief The callback for display status changes.
 */
@property (nonatomic, copy) DisplayStatusCallback displayStatusCallback;

/**
 * @brief The callback for alert status changes.
 */
@property (nonatomic, copy) AlertStatusCallback alertStatusCallback;

@end

