//
//  OdvMonitor.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/4/14.
//

#ifndef OdvMonitor_h
#define OdvMonitor_h


#endif /* OdvMonitor_h */

#import <Foundation/Foundation.h>
#import <UniPRT/TcpComm.h>
#import <UniPRT/JsonComm.h>
#import <UniPRT/OdvReport.h>

/**
 * \~English
 * @class OdvMonitor
 * @brief Provide support for ODV related services. Module to facilitate listening/retrieval of printer unsolicited messaging related to ODV.
 *
 * Unsolicited messages, if enabled on printer, can be sent at any time. This module requires a constant "status" connection to the printer's management port in order to listen for any messaging sent out by printer.
 *
 * @remarks
 * - For ODV reports to be sent out by printer, the following settings must be present:
 *   - On printer OVD menus: "Validator Active = Enable", "Telemetry Path = Network Port", "Telemetry Data = Full Report"
 * @ingroup Monitor
 *
 * \~Chinese
 * @class OdvMonitor
 * @brief 提供与 ODV 相关的服务支持。模块用于监听/检索与 ODV 相关的打印机非请求消息。
 *
 * 如果在打印机上启用了非请求消息，则可以随时发送。该模块需要与打印机的管理端口保持恒定的“状态”连接，以便监听打印机发送的任何消息。
 *
 * @remarks
 * - 要使打印机发送 ODV 报告，必须满足以下设置：
 *   - 在打印机 OVD 菜单中："Validator Active = Enable", "Telemetry Path = Network Port", "Telemetry Data = Full Report"
 * @ingroup Monitor
 *
 * \~Chinese-Traditional
 * @class OdvMonitor
 * @brief 提供與 ODV 相關的服務支持。模塊用於監聽/檢索與 ODV 相關的打印機非請求消息。
 *
 * 如果在打印機上啟用了非請求消息，則可以隨時發送。該模塊需要與打印機的管理端口保持恆定的“狀態”連接，以便監聽打印機發送的任何消息。
 *
 * @remarks
 * - 要使打印機發送 ODV 報告，必須滿足以下設置：
 *   - 在打印機 OVD 菜單中："Validator Active = Enable", "Telemetry Path = Network Port", "Telemetry Data = Full Report"
 * @ingroup Monitor
 */
@interface OdvMonitor : NSObject

/** @cond */

/**
 * \~English
 * @brief A constant ID that the class does not care about.
 *
 * \~Chinese
 * @brief 类不关心的常量 ID。
 *
 * \~Chinese-Traditional
 * @brief 類不關心的常量 ID。
 */
@property (nonatomic, strong) NSString *DONT_CARE_ID;

/**
 * \~English
 * @brief The thread for listening to ODV reports.
 *
 * \~Chinese
 * @brief 用于监听 ODV 报告的线程。
 *
 * \~Chinese-Traditional
 * @brief 用於監聽 ODV 報告的線程。
 */
@property (nonatomic, strong) NSThread *listenerThread;

/**
 * \~English
 * @brief The synchronization context, assumed to be an NSObject or similar.
 *
 * \~Chinese
 * @brief 同步上下文，假定为 NSObject 或类似对象。
 *
 * \~Chinese-Traditional
 * @brief 同步上下文，假定為 NSObject 或類似對象。
 */
@property (nonatomic, strong) id syncContext;

/**
 * \~English
 * @brief The descriptor managed by the monitor.
 *
 * \~Chinese
 * @brief 由监视器管理的描述符。
 *
 * \~Chinese-Traditional
 * @brief 由監視器管理的描述符。
 */
@property (nonatomic, strong) NSString *managedDescriptor;

/** @endcond */

/**
 * \~English
 * @brief Enable/disable listening/parsing unsolicited ODV reports sent from printer.
 *
 * Note that this must be enabled in order to receive any notifications (OdvReportCallback) when barcode reports are received.
 *
 * \~Chinese
 * @brief 启用/禁用监听/解析打印机发送的非请求 ODV 报告。
 *
 * 请注意，必须启用此功能才能在接收到条码报告时收到任何通知（OdvReportCallback）。
 *
 * \~Chinese-Traditional
 * @brief 啟用/禁用監聽/解析打印機發送的非請求 ODV 報告。
 *
 * 請注意，必須啟用此功能才能在接收到條碼報告時收到任何通知（OdvReportCallback）。
 */
@property (nonatomic, assign) BOOL odvReportListening;

/**
 * \~English
 * @brief Initializes a new instance with the given TCP communication object.
 *
 * @param connection The TCP communication object.
 * @return An initialized instance of OdvMonitor, or nil if the object could not be created.
 *
 * \~Chinese
 * @brief 使用给定的 TCP 通信对象初始化一个新实例。
 *
 * @param connection TCP 通信对象。
 * @return 一个初始化的 OdvMonitor 实例，如果无法创建对象，则返回 nil。
 *
 * \~Chinese-Traditional
 * @brief 使用給定的 TCP 通信對象初始化一個新實例。
 *
 * @param connection TCP 通信對象。
 * @return 一個初始化的 OdvMonitor 實例，如果無法創建對象，則返回 nil。
 */
- (instancetype)initWithTcpComm:(TcpComm *)connection;

/**
 * \~English
 * @brief Initializes a new instance with the given JSON communication object.
 *
 * @param mgmtComm The JSON communication object.
 * @return An initialized instance of OdvMonitor, or nil if the object could not be created.
 *
 * \~Chinese
 * @brief 使用给定的 JSON 通信对象初始化一个新实例。
 *
 * @param mgmtComm JSON 通信对象。
 * @return 一个初始化的 OdvMonitor 实例，如果无法创建对象，则返回 nil。
 *
 * \~Chinese-Traditional
 * @brief 使用給定的 JSON 通信對象初始化一個新實例。
 *
 * @param mgmtComm JSON 通信對象。
 * @return 一個初始化的 OdvMonitor 實例，如果無法創建對象，則返回 nil。
 */
- (instancetype)initWithJsonComm:(JsonComm *)mgmtComm;

/**
 * \~English
 * @brief Constructor called when creating a connection managed by the SDK.
 *
 * commDescriptor is descriptor for the communication used to connect with printer.
 *
 * TCP descriptor format: "ip" or "ip:port"
 *
 * e.g. "127.0.0.1" or "127.0.0.1:3007"
 *
 * If port is not provided, default value is used.
 *
 * @param commDescriptor The communication descriptor.
 * @return An initialized instance of OdvMonitor, or nil if the object could not be created.
 *
 * \~Chinese
 * @brief 创建由 SDK 管理的连接时调用的构造函数。
 *
 * commDescriptor 是用于连接打印机的通信描述符。
 *
 * TCP 描述符格式："ip" 或 "ip:port"
 *
 * 例如 "127.0.0.1" 或 "127.0.0.1:3007"
 *
 * 如果未提供端口，则使用默认值。
 *
 * @param commDescriptor 通信描述符。
 * @return 一个初始化的 OdvMonitor 实例，如果无法创建对象，则返回 nil。
 *
 * \~Chinese-Traditional
 * @brief 創建由 SDK 管理的連接時調用的構造函數。
 *
 * commDescriptor 是用於連接打印機的通信描述符。
 *
 * TCP 描述符格式："ip" 或 "ip:port"
 *
 * 例如 "127.0.0.1" 或 "127.0.0.1:3007"
 *
 * 如果未提供端口，則使用默認值。
 *
 * @param commDescriptor 通信描述符。
 * @return 一個初始化的 OdvMonitor 實例，如果無法創建對象，則返回 nil。
 */
- (instancetype)initWithCommDescriptor:(NSString *)commDescriptor;

/**
 * \~English
 * @brief Sets whether ODV report listening is enabled.
 *
 * @param value YES to enable listening, NO to disable.
 *
 * \~Chinese
 * @brief 设置是否启用 ODV 报告监听。
 *
 * @param value YES 表示启用监听，NO 表示禁用。
 *
 * \~Chinese-Traditional
 * @brief 設置是否啟用 ODV 報告監聽。
 *
 * @param value YES 表示啟用監聽，NO 表示禁用。
 */
- (void)SetOdvReportListening:(BOOL)value;

/**
 * \~English
 * @typedef OdvReportCallback
 * @brief A callback type for ODV report changes.
 *
 * @param report The new ODV report.
 *
 * \~Chinese
 * @typedef OdvReportCallback
 * @brief ODV 报告更改的回调类型。
 *
 * @param report 新的 ODV 报告。
 *
 * \~Chinese-Traditional
 * @typedef OdvReportCallback
 * @brief ODV 報告更改的回調類型。
 *
 * @param report 新的 ODV 報告。
 */
typedef void (^OdvReportCallback)(OdvReport *report);

/**
 * \~English
 * @brief Holds the function to call when ODV reports are received. Function must match the signature of OdvReportNotice.
 *
 * The parameter passed into the function is an OdvReport object representing the ODV report received.
 *
 * \~Chinese
 * @brief 用于接收 ODV 报告时调用的函数。函数必须符合 OdvReportNotice 的签名。
 *
 * 传递给函数的参数是一个 OdvReport 对象，表示收到的 ODV 报告。
 *
 * \~Chinese-Traditional
 * @brief 用於接收 ODV 報告時調用的函數。函數必須符合 OdvReportNotice 的簽名。
 *
 * 傳遞給函數的參數是一個 OdvReport 對象，表示收到的 ODV 報告。
 */
@property (nonatomic, copy) OdvReportCallback odvReportCallback;

@end
