from enum import Enum
from typing import Protocol, Optional


class RfidPasswordTypeEnum(Enum):
    """
    Passwords for accessing/securing memory areas on tags that support this feature.
    """
    NONE = 0
    LOCK = 1
    PERMA_LOCK = 2
    UNLOCK = 3
    PERMA_UNLOCK = 4


class RfidMemBlockEnum(Enum):
    """
    RFID memory block selection.
    """
    INVALID_MEM_BLOCK = 0
    EPC = 1
    TID = 2
    USER = 3
    ACCESS_CODE = 4
    KILL_CODE = 5
    PC = 6
    RESERVE = 7


class IRfidBankSelect(Protocol):
    """
    A protocol defining the interface for RfidBankSelect classes.
    """
    memory: RfidMemBlockEnum


class IRfidPassword(Protocol):
    """
    For cases where memory access is password protected.
    """
    password_type: RfidPasswordTypeEnum
    password: str


class IRfidBitField(Protocol):
    """
    A protocol defining the interface for RfidWrite classes.
    """
    offset_from_start: int
    data: str


class IRfidWrite(Protocol):
    """
    Write to different areas or memory blocks of an RFID tag.
    """
    memory: RfidMemBlockEnum
    password: Optional[IRfidPassword]
    data: IRfidBitField


class ARfidWrite(IRfidBankSelect, IRfidBitField, IRfidPassword):
    """
    Base class for ARfidWrite.
    """
    memory: RfidMemBlockEnum
    password_type: RfidPasswordTypeEnum
    password: str
    offset_from_start: int
    data: str

    def __init__(self, mem_block: RfidMemBlockEnum = RfidMemBlockEnum.EPC, data: str = ""):
        self.memory = mem_block
        self.password_type = RfidPasswordTypeEnum.NONE
        self.password = ""
        self.data = data
        self.offset_from_start = 0

    def __str__(self) -> str:
        """
        Returns a string representation of the RFID write operation.
        """
        raise NotImplementedError("Subclasses should implement this method")
