//
//  BtComm.h
//  TSCPrinters SDK
//
//  Created by Realbuber on 2024/3/26.
//
/**@cond**/

#import <Foundation/Foundation.h>
#import <IOBluetooth/IOBluetooth.h>
#import <UniPRT/AComm.h>

/**
 * \~English
 * @class BtComm
 * @brief A class for Bluetooth communication, inheriting from AComm.
 * @ingroup Comm
 * \~Chinese
 * @class BtComm
 * @brief 蓝牙通信类，继承自 AComm。
 * @ingroup Comm
 * \~Chinese-Traditional
 * @class BtComm
 * @brief 藍牙通信類，繼承自 AComm。
 * @ingroup Comm
 */
@interface BtComm : AComm

/**
 * \~English
 * @brief The data received from the device.
 * \~Chinese
 * @brief 从设备接收的数据。
 * \~Chinese-Traditional
 * @brief 從設備接收的數據。
 */
@property (nonatomic, strong) NSMutableData *receivedData;

/**
 * \~English
 * @brief Initializes a new instance with the given device address.
 *
 * @param deviceAddress The Bluetooth address of the device.
 * @return An initialized instance of BtComm, or nil if the object could not be created.
 * \~Chinese
 * @brief 使用给定的设备地址初始化一个新实例。
 *
 * @param deviceAddress 设备的蓝牙地址。
 * @return 初始化后的 BtComm 实例，如果无法创建则返回 nil。
 * \~Chinese-Traditional
 * @brief 使用給定的設備地址初始化一個新實例。
 *
 * @param deviceAddress 設備的藍牙地址。
 * @return 初始化後的 BtComm 實例，如果無法創建則返回 nil。
 */
- (instancetype)initWithDeviceAddress:(NSString *)deviceAddress;

/**
 * \~English
 * @brief Opens the connection to the Bluetooth device.
 * \~Chinese
 * @brief 打开与蓝牙设备的连接。
 * \~Chinese-Traditional
 * @brief 打開與藍牙設備的連接。
 */
- (void)open;

/**
 * \~English
 * @brief Closes the connection to the Bluetooth device.
 * \~Chinese
 *  * @brief 关闭与蓝牙设备的连接。
 * \~Chinese-Traditional
 * @brief 關閉與藍牙設備的連接。
 */
- (void)close;

/**
 * \~English
 * @brief Writes data to the Bluetooth device.
 *
 * @param text The data to write to the device.
 * \~Chinese
 * @brief 向蓝牙设备写入数据。
 *
 * @param text 要写入设备的数据。
 * \~Chinese-Traditional
 * @brief 向藍牙設備寫入數據。
 *
 * @param text 要寫入設備的數據。
 */
- (void)write:(NSData *)text;

/**
 * \~English
 * @brief Reads data from the Bluetooth device.
 *
 * @return The data read from the device.
 * \~Chinese
 * @brief 从蓝牙设备读取数据。
 *
 * @return 从设备读取的数据。
 * \~Chinese-Traditional
 * @brief 從藍牙設備讀取數據。
 *
 * @return 從設備讀取的數據。
 */
- (NSData *)read;

/**
 * \~English
 * @brief Writes data to the Bluetooth device and waits for a response.
 *
 * @param text The data to write to the device.
 * @param responseStartTimeOut The timeout interval for the start of the response.
 * @param responseEndTimeOut The timeout interval for the end of the response.
 * @param completionToken A token indicating the completion of the response.
 * @return The data received in response from the device.
 * \~Chinese
 * @brief 向蓝牙设备写入数据并等待响应。
 *
 * @param text 要写入设备的数据。
 * @param responseStartTimeOut 响应开始的超时时间。
 * @param responseEndTimeOut 响应结束的超时时间。
 * @param completionToken 表示响应完成的标记。
 * @return 从设备收到的响应数据。
 * \~Chinese-Traditional
 * @brief 向藍牙設備寫入數據並等待響應。
 *
 * @param text 要寫入設備的數據。
 * @param responseStartTimeOut 響應開始的超時時間。
 * @param responseEndTimeOut 響應結束的超時時間。
 * @param completionToken 表示響應完成的標記。
 * @return 從設備收到的響應數據。
 */
- (NSData *)writeAndWaitForResponse:(NSData *)text
                 responseStartTimeOut:(NSTimeInterval)responseStartTimeOut
                   responseEndTimeOut:(NSTimeInterval)responseEndTimeOut
                      completionToken:(NSString *)completionToken;

@end

/**@endcond**/
